/*
 * Copyright (c) 2006 Oracle.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
#include <linux/kernel.h>

#include "rds.h"
#include "ib.h"

/*
 * This only happens on shutdown.
 */
DECLARE_WAIT_QUEUE_HEAD(rds_ib_ring_empty_wait);

void rds_ib_ring_init(struct rds_ib_work_ring *ring, u32 nr)
{
	/* XXX lockdep class */
	spin_lock_init(&ring->w_lock);
	ring->w_nr = nr;
	ring->w_nr_free = nr;
	ring->w_next_free = 0;
	rdsdebug("ring %p nr %u\n", ring, ring->w_nr);
}

u32 rds_ib_ring_alloc(struct rds_ib_work_ring *ring, u32 val, u32 *pos)
{
	unsigned long flags;
	u32 ret = 0;

	spin_lock_irqsave(&ring->w_lock, flags);

	rdsdebug("ring %p val %u next %u free %u\n", ring, val,
		 ring->w_next_free, ring->w_nr_free);

	if (val && ring->w_nr_free) {
		ret = min(val, ring->w_nr_free);
		*pos = ring->w_next_free;

		ring->w_nr_free -= ret;
		ring->w_next_free += ret;
		if (ring->w_next_free >= ring->w_nr)
			ring->w_next_free -= ring->w_nr;
	}

	spin_unlock_irqrestore(&ring->w_lock, flags);

	return ret;
}

static int rds_ib_ring_empty_locked(struct rds_ib_work_ring *ring)
{
	assert_spin_locked(&ring->w_lock);
	rdsdebug("ring %p nr %d free %u\n", ring, ring->w_nr, ring->w_nr_free);
	return ring->w_nr_free == ring->w_nr;
}

static u32 rds_ib_ring_free_locked(struct rds_ib_work_ring *ring, u32 val)
{
	assert_spin_locked(&ring->w_lock);

	BUG_ON(ring->w_nr_free + val > ring->w_nr);
	ring->w_nr_free += val;

	if (rds_ib_ring_empty_locked(ring) &&
	    waitqueue_active(&rds_ib_ring_empty_wait))
		wake_up(&rds_ib_ring_empty_wait);

	return ring->w_nr_free;
}

u32 rds_ib_ring_free(struct rds_ib_work_ring *ring, u32 val)
{
	unsigned long flags;
	u32 ret;

	spin_lock_irqsave(&ring->w_lock, flags);
	rdsdebug("ring %p val %u free %u\n", ring, val, ring->w_nr_free);
	ret = rds_ib_ring_free_locked(ring, val);
	spin_unlock_irqrestore(&ring->w_lock, flags);
	return ret;
}

void rds_ib_ring_unalloc(struct rds_ib_work_ring *ring, u32 val)
{
	unsigned long flags;

	spin_lock_irqsave(&ring->w_lock, flags);
	rdsdebug("ring %p val %u next %u free %u\n", ring, val,
		 ring->w_next_free, ring->w_nr_free);
	rds_ib_ring_free_locked(ring, val);
	ring->w_next_free -= val;
	if (ring->w_next_free > ring->w_nr)
		ring->w_next_free += ring->w_nr;
	spin_unlock_irqrestore(&ring->w_lock, flags);
}

int rds_ib_ring_empty(struct rds_ib_work_ring *ring)
{
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&ring->w_lock, flags);
	ret = rds_ib_ring_empty_locked(ring);
	spin_unlock_irqrestore(&ring->w_lock, flags);

	return ret;
}

/*
 * returns the oldest alloced ring entry.  This will be the next one
 * freed.  This can't be called if there are none allocated.
 */
u32 rds_ib_ring_oldest(struct rds_ib_work_ring *ring)
{
	unsigned long flags;
	u32 ret;

	spin_lock_irqsave(&ring->w_lock, flags);
	BUG_ON(ring->w_nr_free == ring->w_nr);
	ret = (ring->w_next_free + ring->w_nr_free) % ring->w_nr;
	rdsdebug("ring %p ret %u next %u free %u\n", ring, ret,
		 ring->w_next_free, ring->w_nr_free);
	spin_unlock_irqrestore(&ring->w_lock, flags);

	return ret;
}

/*
 * returns the number of completed work requests.
 */

u32 rds_ib_ring_completed(struct rds_ib_work_ring *ring, u32 wr_id, u32 oldest)
{
	unsigned long flags;
	u32 ret;

	spin_lock_irqsave(&ring->w_lock, flags);
	if (oldest <= (unsigned long long)wr_id)
		ret = (unsigned long long)wr_id - oldest + 1;
	else
		ret = ring->w_nr - oldest + (unsigned long long)wr_id + 1;

	rdsdebug("ring %p ret %u wr_id %u oldest %u\n", ring, ret,
		 wr_id, oldest);
	spin_unlock_irqrestore(&ring->w_lock, flags);

	return ret;
}
